﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/globalaccelerator/GlobalAcceleratorRequest.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>

#include <utility>

namespace Aws {
namespace GlobalAccelerator {
namespace Model {

/**
 */
class AdvertiseByoipCidrRequest : public GlobalAcceleratorRequest {
 public:
  AWS_GLOBALACCELERATOR_API AdvertiseByoipCidrRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AdvertiseByoipCidr"; }

  AWS_GLOBALACCELERATOR_API Aws::String SerializePayload() const override;

  AWS_GLOBALACCELERATOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The address range, in CIDR notation. This must be the exact range that you
   * provisioned. You can't advertise only a portion of the provisioned range.</p>
   * <p> For more information, see <a
   * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring
   * your own IP addresses (BYOIP)</a> in the Global Accelerator Developer Guide.</p>
   */
  inline const Aws::String& GetCidr() const { return m_cidr; }
  inline bool CidrHasBeenSet() const { return m_cidrHasBeenSet; }
  template <typename CidrT = Aws::String>
  void SetCidr(CidrT&& value) {
    m_cidrHasBeenSet = true;
    m_cidr = std::forward<CidrT>(value);
  }
  template <typename CidrT = Aws::String>
  AdvertiseByoipCidrRequest& WithCidr(CidrT&& value) {
    SetCidr(std::forward<CidrT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cidr;
  bool m_cidrHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
